<?php
/**
 * WORKING UPLOAD API - SIMPLIFIED VERSION
 * Optimized for 72 DPI web performance with database integration
 */

header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, DELETE, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization');

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit();
}

// Configuration
$UPLOAD_DIR = 'uploads/';
$THUMBS_DIR = 'uploads/thumbs/';
$MAX_IMAGE_SIZE = 5242880; // 5MB for images
$MAX_DOCUMENT_SIZE = 10485760; // 10MB for documents
$MIN_IMAGE_WIDTH = 300;
$MIN_IMAGE_HEIGHT = 300;
$THUMB_HEIGHT = 150;
$THUMB_QUALITY = 75;  // Optimized for web
$WEB_QUALITY = 78;    // Main image quality

// Allowed file types
$ALLOWED_IMAGE_EXTENSIONS = ['jpg', 'jpeg', 'png', 'gif', 'webp'];
$ALLOWED_DOCUMENT_EXTENSIONS = ['pdf', 'doc', 'docx', 'txt', 'rtf', 'odt'];
$ALLOWED_EXTENSIONS = array_merge($ALLOWED_IMAGE_EXTENSIONS, $ALLOWED_DOCUMENT_EXTENSIONS);

$ALLOWED_IMAGE_MIME_TYPES = [
    'image/jpeg', 'image/jpg', 'image/png', 'image/gif', 'image/webp'
];

$ALLOWED_DOCUMENT_MIME_TYPES = [
    'application/pdf', 'application/msword',
    'application/vnd.openxmlformats-officedocument.wordprocessingml.document',
    'text/plain', 'text/rtf', 'application/rtf',
    'application/vnd.oasis.opendocument.text'
];

$ALLOWED_MIME_TYPES = array_merge($ALLOWED_IMAGE_MIME_TYPES, $ALLOWED_DOCUMENT_MIME_TYPES);

// Database connection
function getDbConnection() {
    $host = 'localhost';
    $username = 'root';
    $password = '';
    $database = 'pictures_gallery';
    
    $conn = new mysqli($host, $username, $password, $database);
    
    if ($conn->connect_error) {
        throw new Exception('Database connection failed: ' . $conn->connect_error);
    }
    
    $conn->set_charset("utf8");
    return $conn;
}

// Get available database columns
function getAvailableColumns($conn) {
    $result = $conn->query("DESCRIBE pictures_gallery");
    $columns = [];
    while ($row = $result->fetch_assoc()) {
        $columns[] = $row['Field'];
    }
    return $columns;
}

try {
    if ($_SERVER['REQUEST_METHOD'] === 'POST') {
        // HANDLE FILE UPLOAD
        if (!isset($_FILES['file'])) {
            throw new Exception('No file uploaded');
        }
        
        $file = $_FILES['file'];
        $filename = $_POST['filename'] ?? $file['name'];
        $title = $_POST['title'] ?? pathinfo($filename, PATHINFO_FILENAME);
        $description = $_POST['description'] ?? '';
        
        // Validate file
        if ($file['error'] !== UPLOAD_ERR_OK) {
            throw new Exception('Upload error: ' . $file['error']);
        }
        
        // Get file info
        $file_extension = strtolower(pathinfo($filename, PATHINFO_EXTENSION));
        $finfo = finfo_open(FILEINFO_MIME_TYPE);
        $mime_type = finfo_file($finfo, $file['tmp_name']);
        finfo_close($finfo);
        
        // Determine file type
        $is_image = in_array($file_extension, $ALLOWED_IMAGE_EXTENSIONS) && 
                   in_array($mime_type, $ALLOWED_IMAGE_MIME_TYPES);
        $is_document = in_array($file_extension, $ALLOWED_DOCUMENT_EXTENSIONS) && 
                      in_array($mime_type, $ALLOWED_DOCUMENT_MIME_TYPES);
        
        if (!$is_image && !$is_document) {
            throw new Exception('Invalid file type. Allowed types: ' . implode(', ', $ALLOWED_EXTENSIONS));
        }
        
        // Check file size based on type
        if ($is_image && $file['size'] > $MAX_IMAGE_SIZE) {
            throw new Exception('Image size exceeds maximum allowed size of ' . ($MAX_IMAGE_SIZE / 1024 / 1024) . 'MB');
        } elseif ($is_document && $file['size'] > $MAX_DOCUMENT_SIZE) {
            throw new Exception('Document size exceeds maximum allowed size of ' . ($MAX_DOCUMENT_SIZE / 1024 / 1024) . 'MB');
        }
        
        // Security checks
        $dangerous_extensions = ['php', 'phtml', 'php3', 'php4', 'php5', 'exe', 'bat', 'com', 'scr', 'js', 'jar'];
        if (in_array($file_extension, $dangerous_extensions)) {
            throw new Exception('File type is not allowed for security reasons');
        }
        
        // Image-specific validation
        if ($is_image) {
            $image_info = getimagesize($file['tmp_name']);
            if ($image_info === false) {
                throw new Exception('Invalid image file');
            }
            
            $width = $image_info[0];
            $height = $image_info[1];
            
            if ($width < $MIN_IMAGE_WIDTH || $height < $MIN_IMAGE_HEIGHT) {
                throw new Exception('Image dimensions too small. Minimum size: ' . $MIN_IMAGE_WIDTH . 'x' . $MIN_IMAGE_HEIGHT . ' pixels');
            }
        }
        
        // Create upload directory if it doesn't exist
        if (!file_exists($UPLOAD_DIR)) {
            mkdir($UPLOAD_DIR, 0755, true);
        }
        
        // Generate safe filename
        $safe_filename = preg_replace('/[^a-zA-Z0-9._-]/', '_', $filename);
        $unique_filename = pathinfo($safe_filename, PATHINFO_FILENAME) . '_' . uniqid() . '_' . time() . '.' . pathinfo($safe_filename, PATHINFO_EXTENSION);
        $upload_path = $UPLOAD_DIR . $unique_filename;
        
        // Move uploaded file
        if (!move_uploaded_file($file['tmp_name'], $upload_path)) {
            throw new Exception('Failed to save uploaded file');
        }
        
        $response = [
            'success' => true,
            'message' => 'File uploaded successfully',
            'path' => '/' . $upload_path,
            'url' => 'https://' . $_SERVER['HTTP_HOST'] . '/' . $upload_path,
            'filename' => basename($upload_path),
            'size' => filesize($upload_path),
            'type' => $is_image ? 'image' : 'document'
        ];
        
        // Initialize optimization variables
        $thumbnail_path = null;
        $thumbnail_url = null;
        $webp_path = null;
        $webp_url = null;
        $webp_size = null;
        $image_width = null;
        $image_height = null;
        $original_size = $file['size'];
        $final_size = filesize($upload_path);
        $compression_ratio = null;
        $is_optimized = 0;
        
        // Process image if it's an image file
        if ($is_image && extension_loaded('gd')) {
            $image_info = getimagesize($upload_path);
            $image_width = $image_info[0];
            $image_height = $image_info[1];
            $image_type = $image_info[2];
            
            // Calculate compression ratio
            if ($original_size > 0 && $final_size > 0) {
                $compression_ratio = round((($original_size - $final_size) / $original_size) * 100, 2);
            }
            
            // Create optimized version if image is too large
            $max_web_width = 1920;
            $max_web_height = 1080;
            
            if ($image_width > $max_web_width || $image_height > $max_web_height) {
                // Create optimized version
                $source_image = null;
                switch ($image_type) {
                    case IMAGETYPE_JPEG:
                        $source_image = imagecreatefromjpeg($upload_path);
                        break;
                    case IMAGETYPE_PNG:
                        $source_image = imagecreatefrompng($upload_path);
                        break;
                    case IMAGETYPE_GIF:
                        $source_image = imagecreatefromgif($upload_path);
                        break;
                    case IMAGETYPE_WEBP:
                        $source_image = imagecreatefromwebp($upload_path);
                        break;
                }
                
                if ($source_image) {
                    // Calculate new dimensions
                    $aspect_ratio = $image_width / $image_height;
                    if ($image_width > $image_height) {
                        $new_width = $max_web_width;
                        $new_height = round($new_width / $aspect_ratio);
                        if ($new_height > $max_web_height) {
                            $new_height = $max_web_height;
                            $new_width = round($new_height * $aspect_ratio);
                        }
                    } else {
                        $new_height = $max_web_height;
                        $new_width = round($new_height * $aspect_ratio);
                        if ($new_width > $max_web_width) {
                            $new_width = $max_web_width;
                            $new_height = round($new_width / $aspect_ratio);
                        }
                    }
                    
                    // Create optimized image
                    $optimized_image = imagecreatetruecolor($new_width, $new_height);
                    imagealphablending($optimized_image, false);
                    imagesavealpha($optimized_image, true);
                    $transparent = imagecolorallocatealpha($optimized_image, 255, 255, 255, 127);
                    imagefilledrectangle($optimized_image, 0, 0, $new_width, $new_height, $transparent);
                    
                    imagecopyresampled($optimized_image, $source_image, 0, 0, 0, 0, $new_width, $new_height, $image_width, $image_height);
                    
                    // Save optimized version
                    $optimized_filename = str_replace('.', '_optimized.', $unique_filename);
                    $optimized_path = $UPLOAD_DIR . $optimized_filename;
                    
                    switch ($image_type) {
                        case IMAGETYPE_JPEG:
                            if (function_exists('imageinterlace')) {
                                imageinterlace($optimized_image, true); // Progressive JPEG
                            }
                            imagejpeg($optimized_image, $optimized_path, $WEB_QUALITY);
                            break;
                        case IMAGETYPE_PNG:
                            imagepng($optimized_image, $optimized_path, 7);
                            break;
                        case IMAGETYPE_GIF:
                            imagegif($optimized_image, $optimized_path);
                            break;
                        case IMAGETYPE_WEBP:
                            imagewebp($optimized_image, $optimized_path, $WEB_QUALITY);
                            break;
                    }
                    
                    // Update response with optimized version
                    $response['optimized'] = [
                        'path' => '/' . $optimized_path,
                        'url' => 'https://' . $_SERVER['HTTP_HOST'] . '/' . $optimized_path,
                        'width' => $new_width,
                        'height' => $new_height,
                        'size' => filesize($optimized_path)
                    ];
                    
                    // Update dimensions
                    $image_width = $new_width;
                    $image_height = $new_height;
                    $upload_path = $optimized_path; // Use optimized version as main
                    $final_size = filesize($optimized_path);
                    $compression_ratio = round((($original_size - $final_size) / $original_size) * 100, 2);
                    $is_optimized = 1;
                    
                    imagedestroy($optimized_image);
                    imagedestroy($source_image);
                }
            }
            
            // Create WebP version
            if (function_exists('imagewebp') && $image_type != IMAGETYPE_WEBP) {
                $source_image = null;
                switch ($image_type) {
                    case IMAGETYPE_JPEG:
                        $source_image = imagecreatefromjpeg($upload_path);
                        break;
                    case IMAGETYPE_PNG:
                        $source_image = imagecreatefrompng($upload_path);
                        break;
                    case IMAGETYPE_GIF:
                        $source_image = imagecreatefromgif($upload_path);
                        break;
                }
                
                if ($source_image) {
                    $webp_filename = str_replace('.' . $file_extension, '.webp', $unique_filename);
                    $webp_path = $UPLOAD_DIR . $webp_filename;
                    
                    if (imagewebp($source_image, $webp_path, 80)) {
                        $webp_size = filesize($webp_path);
                        $webp_url = 'https://' . $_SERVER['HTTP_HOST'] . '/' . $webp_path;
                        $response['webp'] = [
                            'path' => '/' . $webp_path,
                            'url' => $webp_url,
                            'size' => $webp_size
                        ];
                    }
                    
                    imagedestroy($source_image);
                }
            }
            
            // Create thumbnail
            if (!file_exists($THUMBS_DIR)) {
                mkdir($THUMBS_DIR, 0755, true);
            }
            
            $source_image = null;
            switch ($image_type) {
                case IMAGETYPE_JPEG:
                    $source_image = imagecreatefromjpeg($upload_path);
                    break;
                case IMAGETYPE_PNG:
                    $source_image = imagecreatefrompng($upload_path);
                    break;
                case IMAGETYPE_GIF:
                    $source_image = imagecreatefromgif($upload_path);
                    break;
                case IMAGETYPE_WEBP:
                    $source_image = imagecreatefromwebp($upload_path);
                    break;
            }
            
            if ($source_image) {
                // Calculate thumbnail dimensions
                $aspect_ratio = $image_width / $image_height;
                $thumb_width = round($THUMB_HEIGHT * $aspect_ratio);
                
                // Create thumbnail
                $thumbnail = imagecreatetruecolor($thumb_width, $THUMB_HEIGHT);
                imagealphablending($thumbnail, false);
                imagesavealpha($thumbnail, true);
                $transparent = imagecolorallocatealpha($thumbnail, 255, 255, 255, 127);
                imagefilledrectangle($thumbnail, 0, 0, $thumb_width, $THUMB_HEIGHT, $transparent);
                
                imagecopyresampled($thumbnail, $source_image, 0, 0, 0, 0, $thumb_width, $THUMB_HEIGHT, $image_width, $image_height);
                
                // Save thumbnail
                $thumb_filename = str_replace('.', '_thumb.', $unique_filename);
                $thumbnail_path = $THUMBS_DIR . $thumb_filename;
                
                switch ($image_type) {
                    case IMAGETYPE_JPEG:
                        imagejpeg($thumbnail, $thumbnail_path, $THUMB_QUALITY);
                        break;
                    case IMAGETYPE_PNG:
                        imagepng($thumbnail, $thumbnail_path, 6);
                        break;
                    case IMAGETYPE_GIF:
                        imagegif($thumbnail, $thumbnail_path);
                        break;
                    case IMAGETYPE_WEBP:
                        imagewebp($thumbnail, $thumbnail_path, $THUMB_QUALITY);
                        break;
                }
                
                if (file_exists($thumbnail_path)) {
                    $thumbnail_url = 'https://' . $_SERVER['HTTP_HOST'] . '/' . $thumbnail_path;
                    $response['thumbnail'] = [
                        'path' => '/' . $thumbnail_path,
                        'url' => $thumbnail_url,
                        'width' => $thumb_width,
                        'height' => $THUMB_HEIGHT,
                        'size' => filesize($thumbnail_path)
                    ];
                }
                
                imagedestroy($thumbnail);
                imagedestroy($source_image);
            }
        }
        
        // Save to database
        try {
            $conn = getDbConnection();
            $available_columns = getAvailableColumns($conn);
            
            // Build insert data dynamically based on available columns
            $insert_data = [
                'title' => $title,
                'description' => $description,
                'local_path' => $upload_path,
                'remote_path' => '/' . $upload_path,
                'remote_url' => $response['url'],
                'file_name' => $unique_filename,
                'file_size' => $final_size,
                'file_type' => $mime_type
            ];
            
            // Add optimization columns if they exist
            $optional_data = [
                'file_type_category' => $is_image ? 'image' : 'document',
                'image_width' => $image_width,
                'image_height' => $image_height,
                'webp_path' => $webp_path,
                'webp_url' => $webp_url,
                'webp_size' => $webp_size,
                'thumbnail_path' => $thumbnail_path,
                'thumbnail_url' => $thumbnail_url,
                'thumbnail_medium_path' => $thumbnail_path,
                'thumbnail_medium_url' => $thumbnail_url,
                'original_size' => $original_size,
                'compression_ratio' => $compression_ratio,
                'is_optimized' => $is_optimized
            ];
            
            foreach ($optional_data as $column => $value) {
                if (in_array($column, $available_columns)) {
                    $insert_data[$column] = $value;
                }
            }
            
            // Build SQL dynamically
            $columns = array_keys($insert_data);
            $placeholders = array_fill(0, count($columns), '?');
            $sql = "INSERT INTO pictures_gallery (" . implode(', ', $columns) . ") VALUES (" . implode(', ', $placeholders) . ")";
            
            $stmt = $conn->prepare($sql);
            
            // Build parameter types and values
            $param_types = '';
            $param_values = [];
            
            foreach ($insert_data as $value) {
                if (is_int($value)) {
                    $param_types .= 'i';
                } elseif (is_float($value)) {
                    $param_types .= 'd';
                } elseif (is_null($value)) {
                    $param_types .= 's';
                } else {
                    $param_types .= 's';
                }
                $param_values[] = $value;
            }
            
            $stmt->bind_param($param_types, ...$param_values);
            
            if ($stmt->execute()) {
                $image_id = $conn->insert_id;
                $response['database'] = [
                    'success' => true,
                    'image_id' => $image_id,
                    'message' => 'File information saved to database'
                ];
                $response['image_id'] = $image_id;
            } else {
                $response['database'] = [
                    'success' => false,
                    'message' => 'Database save failed: ' . $stmt->error
                ];
            }
            
            $stmt->close();
            $conn->close();
            
        } catch (Exception $e) {
            $response['database'] = [
                'success' => false,
                'message' => 'Database error: ' . $e->getMessage()
            ];
        }
        
        // Add optimization info to response
        $response['optimization'] = [
            'is_optimized' => $is_optimized,
            'compression_ratio' => $compression_ratio,
            'original_size' => $original_size,
            'final_size' => $final_size,
            'webp_available' => !empty($webp_path),
            'thumbnail_created' => !empty($thumbnail_path)
        ];
        
        echo json_encode($response);
        
    } else {
        throw new Exception('Method not allowed. Use POST for upload.');
    }
    
} catch (Exception $e) {
    http_response_code(400);
    echo json_encode([
        'success' => false, 
        'message' => $e->getMessage()
    ]);
} catch (Error $e) {
    http_response_code(500);
    echo json_encode([
        'success' => false, 
        'message' => 'Server error: ' . $e->getMessage()
    ]);
}
?>
